﻿using System;
using Microsoft.VisualBasic;
using System.Windows.Forms;
using System.Globalization;

namespace MVDataServer {

	/// <summary>
	/// The Help class contains supporting routines for many of the options available under the Help Menu
	/// </summary>
	internal class Help {

		private static readonly string NL = "\r\n";
		private static readonly string GAP = "\r\n\r\n";

		/// <summary>
		/// Set the Date Format used to display datestamps (using C# date formating conventions)
		/// </summary>
		/// <param name="main">The Main form that invokes this function</param>
		internal static void SetDateFormat(Main main) {
			string dateFormat = MVShareStore.Data.DateFormat;
			bool done = false;

			while (!done) {
				dateFormat = Interaction.InputBox("Enter the string defining the C# Date / Time Format you want to use when displaying DateStamps", "Set / Change Date Time Format", dateFormat);
				string current = DateTime.Now.ToString(dateFormat, CultureInfo.CurrentCulture);
				main.setStatusBar("Current Date / Time: " + current);
				DialogResult result = MessageBox.Show("Current Date / Time: " + current + ". Are you happy with this date / time format?", "Set Date / Time Format", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
				if (result == DialogResult.Cancel) break;
				else if (result == DialogResult.No) continue;
				else {
					MVShareStore.Data.DateFormat = dateFormat;
					done = true;
				}
			}
		}

		/// <summary>
		/// Set the number of Error Messages to keep in the error history
		/// </summary>
		/// <param name="main"></param>
		internal static void SetErrorHistoryDepth(Main main) {
			string input = Convert.ToString(MVShareStore.Settings.ErrorHistoryDepth);
			input = Interaction.InputBox("How many error messages do you want to keep in the error message history?", "Error Message History", input);
			int depth = 0;
			if (int.TryParse(input, out depth)) {
				MVShareStore.Settings.ErrorHistoryDepth = depth;
				main.setStatusBar("Error message history depth now set to " + depth);
			}
		}

		internal static void ShowErrorMessages(Main main) {
			MVShareStore.Data.ShowListErrorMessagesForm();
		}

		/// <summary>
		/// Show the Help Library Concepts and Design Dialog Box
		/// </summary>
		/// <param name="main">The Main form that invokes this function</param>
		internal static void ShowConcepts(Main main) {
			forms.ShowHelp showHelp = new forms.ShowHelp("MVShareStore Library Concepts and Design features", HelpConcepts);
			showHelp.Show();
		}

		/// <summary>
		/// Show the Help Library supported Data Types Dialog Box
		/// </summary>
		/// <param name="main">The Main form that invokes this function</param>
		internal static void ShowDataTypes(Main main) {
			forms.ShowHelp showHelp = new forms.ShowHelp("MVShareStore Library supported Data Types", HelpDataTypes);
			showHelp.Show();
		}

		/// <summary>
		/// Show the Help API Details Dialog Box
		/// </summary>
		/// <param name="main">The Main form that invokes this function</param>
		internal static void ShowApiDetails(Main main) {
			forms.ShowHelp showHelp = new forms.ShowHelp("MVShareStore Library API overview", HelpApi);
			showHelp.Show();
		}

		/// <summary>
		/// Show the Help ReadMe file Dialog Box
		/// </summary>
		/// <param name="main">The Main form that invokes this function</param>
		internal static void ShowReadme(Main main) {
			forms.ShowHelp showHelp = new forms.ShowHelp("MVShareStore Library Readme file", HelpReadme);
			showHelp.Show();
		}

		/// <summary>
		/// Show the Help About Dialog Box
		/// </summary>
		/// <param name="main">The Main form that invokes this function</param>
		internal static void ShowHelpAbout(Main main) {
			MessageBox.Show(HelpAbout, "About the MVShareStore Library Server");
		}

		private static string HelpConcepts =
			"The MVShareStore Library implements a method of packaging up multiple data items / variables of many different types in a single package (DataPack) that can used internally within an application and / or it can be saved to disk and read back again (DataStore) and / or published into global memory and shared between multiple disparate processes (DataShare)."
			+ GAP
			+ "There are three basic data containing / handling entities (classes) implemented by this library:"
			+ GAP
			+ "DataPack:"
			+ NL + "This is a collection of Data Items of various Data Types such as long, bool and string (see below) and can be thought of as an array containing entries of different data types or a class containing only data."
			+ NL + "These can optionally be named (names must be unique within that DataPack) and Data Items can be queried and found using either their name or their (0 based) index into the 'array'."
			+ NL + "It implements the ability to add new entries to it, edit existing entries, insert entries and delete entries, find entries by name or index and cycle through all the entries. DataPacks can also be merged and sorted"
			+ GAP

			+ "DataPacks can be used in the following ways:"
			+ NL + "\tEasily converted to DataStores or DataShares"
			+ NL + "\tAct as a set of Name / Value pairs"
			+ NL + "\tAct as a composite return value from a function returning multiple variables of different types"
			+ NL + "\tAct as a composite function parameter containing multiple variables of different types"
			+ NL + "\tUsed as a 'state machine' to communicate between a running thread and the process that owns it"
			+ NL + "\tUsed for any internal communication involving sets of disparate variables"
			+ NL + "\tCan be implemented as Data Items and nested in other DataPacks to create hierarchical data structures"
				
			+ GAP
			+ "DataStore:" 
			+ NL + "This is derived from DataPack and therefore inherits all its capabilities. It implements the ability to save the collections of variables it contains (in the DataPack) to files on disk for future retrieval and re-use."
			+ NL + "It can also potentially have other uses such as storing and retrieving program configuration data / settings / state data and almost anything else you can think of."
			+ NL + "It can be converted into a DataShare so that it can be shared between processes."
			+ GAP

			+ "DataShare:"
			+ NL + "This is also derived from DataPack and implements the sharing of these data collections between different processes running on the same host by storing them in RAM using Memory Mapped files."
			+ NL + "For example a Server type application may have some data it wants to share with a Client program or two programs might be performing similar tasks and need to communicate to tell each other what they are doing or update each other."
			+ NL + "Each DataShare must be named and that name must be unique. It also has to belong to an Owner / User / Group and as marked as private (so that is only be opened by another process using the same Owner / User Group)"
			+ NL + "After a DataShare has been created it can be posted to the ShareStore (see below) and will then appear in the list of currently available DataShares and made globally available to other applications / processes to open and manipulate."
			+ GAP

			+ "DataShares have the following capabilities:"
			+ NL + "\tA DataShare can have an Owner which can be used to filter what DataShares are returned by searches.This can also optionally be used to group sets of DataShares together."
			+ NL + "\tThey can be read only so that no pther processes can modify their contents."
			+ NL + "\tThey can be marked as private so that only the Owner can read and access them(there is no real security provided by the Library here)."
			+ NL + "\tAll DataShares are given a unique numeric ID and entered into the Header List which makes them available(by ID or Name) to all other processes using that ShareStore."
			+ NL + "\tIt can be converted into a DataStore so that it can be saved to a file on disk for future re-use."
			+ GAP 

			+ "Note that DataPacks, DataStores and DataShares are effectively interchangable and can easily be converted between the different types (normally using the functions in the Data class)"
			+ GAP
			+ "ShareStores:"
			+ NL + "There is the concept of a ShareStore which is a collection containing all the DataShares globally available to any MVShareStore enabled application / process on a host that has opened that ShareStore."
			+ NL + "Typically each application will only access a single ShareStore but multiple ShareStores having different names can be implemented on the same host if desired (thereby 'hiding' their data from any application that has a different ShareStore open)."
			+ NL + "Each ShareStore can contain one or more DataShares which are collections of Data Items (variables) which can be shared between MVShareStore enabled applications."
			+ NL + "Normally the default ShareStore will be used and this can be handled automatically by the MVShareStore library itself."
			;

		private static string HelpDataTypes =
			"MVShareStore Library currently supports all the C# primitive Data Types (plus DataPacks). These are all defined in the DataType class and encapsulated by the DataItem class."
			+ GAP
			+ "Data Types currently supported:"
			+ NL + "Bool"	   + "\t" + "bool value (True or False)"
			+ NL + "Byte"	   + "\t" + "byte value (unsigned 8 bit integer) values = 0 - 255"
			+ NL + "Char"	   + "\t" + "char (16 bit) values = U+0000 to U+FFFF"
			+ NL + "Short"	   + "\t" + "short (signed 16 bit integer) values = -32,768 to 32,767"
			+ NL + "Int"	   + "\t" + "int (signed 32 bit integer) values = -2,147,483,648 to 2,147,483,647"
			+ NL + "Long"		+ "\t" + "long (signed 64-bit integer) values = -9,223,372,036,854,775,808 to 9,223,372,036,854,775,807"
			+ NL + "Float"		+ "\t" + "float (4 byte numeric) values = ~6-9 digits"
			+ NL + "Double"	+ "\t" + "double (8 byte numeric) values = ~15-17 digits"
			+ NL + "Decimal"	+ "\t" + "decimal (16 byte numeric) values = ~28-29 digits"
			+ NL + "String"	+ "\t" + "string value"
			+ NL + "DataPack"	+ "\t" + "DataPack value"
			+ GAP

			+ "Every DataPack, DataStore or DataShare is basically a collection (array) of DataItems of any of the supported DataTypes. Each entry (variable) can optionally be named or simply accessed by its index in the collection."
			+ "The DataItems in a DataPack, DataStore or DataShare can be added to, edited, inserted and deleted. However they cannot be sorted because that would potentially change their index in the collection"
			+ GAP
			+ "Note: Support for arrays of these DataTypes is expected to be implemented in a future release (and will be identified using an X suffix (for example StringX will be an array of Strings)."
			;

		private static string HelpApi =
			"MVShareStore Library exposes an API which can be used by any program that implements this Library. Examples of how to use this API are contained in the sample MVDataServer and MVDataClient projects."
			+ GAP
			+ "All the public classes, functions and properties are fully commented meaning that the Visual Studio intellisense should be able to assist you with what is available in this Library and how to use it."
			+ GAP
			+ "It should be noted that many of the functions provided by the Library have default values for some of their parameters. This provides considerable flexibilty regarding the use of these functions but care must be taken to ensure that unexpected behaviour is not the result of these defaults."
			+ GAP
			+ "As a general rule, all the high level functions can be access through the static functions and properties implemented by the MVShareStore.Data class (see below)."
			+ GAP
			+ "Data Class API:"
			+ NL + "----------------------------"
			+ NL + "One of the most important properties in the Data class is Data.LastError because this will contain the last error that occurred if any function or operation failed."
			+ NL + "The Data Class also implements the concept of a current DataPack, DataStore and DataShare (which it provides) and will often automatically update these to reflect the Data package instance you are currently working on."
			+ NL + "Some of the internal settings used by the Library (such as the date format used for date display) can be accessed from the Data class as can the last Directory used and the last File referenced"
			+ NL + "The Data class can (and generally should) be used to create, edit, list and generally manipulate DataItems, DataPacks, DataStores and DataShares." 
			+ NL + "It can also be used to show the (fairly rudimentary) forms implemented by the MVShareStore Library (such as Add, List or Edit Data Items or List Data Shares)."
			+ GAP
			+ "However, lower level access and class specific functionality (such as DataItem adding, querying and editing) can be accessed directly using the capabilities of the DataPack, DataStore and DataShare classes."
			+ GAP
			+ "DataShares and ShareLists:"
			+ NL + "------------------------------------------------"
			+ NL + "The ShareList class can be used to get, sort and filter lists of all the DataShares that have been posted in the current ShareStore and are available to be opened etc. Normally this list can contain up to around 5,000 entries but this can be modified if desired by changing the maximum size of the DataShare Name and Owner fields."
			+ GAP
			+ "Each DataShare is identified using a unique numeric ID that is assigned when it is created and, as well as having a Name, each DataShare can have an Owner / User that can be used to group them as well as restrict access to them. If a DataShare is marked as private, only a client using that same Owner / User is allowed to open it."
			;

		private static string HelpReadme =
			"What is it ?" 
			+ NL + "--------------------"
			+ GAP
			+ "MVShareStore is a programmer's tool implemented as a MS Visual Studio C# Library / DLL that can be used by Windows applications to share data programmatically within a single process and between disparate and separate processes running on the same host."
			+ NL + "It can also save this internal or shared data as files to disk which can then be reloaded later if required."
			+ GAP
			+ "Note: A future version will almost certainly be able to pass data between processes on different hosts (security issues permitting)."
			+ GAP
			+ "What is included"
			+ NL + "------------------------------"
			+ GAP
			+ "A (fully commented) 32 bit and / or 64 bit.NET C# Library / DLL that can be referenced in any Visual Studio .NET project to implement its data handling and sharing capability."
			+ NL + "The Visual Studio project for a simple Windows Data Server application that can be used to demonstrate how the library can be utilised in your own projects or used as a basis to implement similar capabilities."
			+ NL + "The Visual Studio project for a simple Windows Data Client application that can be used to demonstrate how the library can be utilised in your own projects or used as a basis to implement similar capabilities."
			+ GAP
			+ "Installation"
			+ NL + "---------------------"
			+ GAP
			+ "The appropriate (32 or 64 bit) MVShareStore.dll library should be copied to a known location where it can be included in your Project's References."
			+ NL + "If required the MVDataServer and / or MVDataClient projects can be copied to a known location and either referenced, used as a basis or included in your own project(s)."
			+ GAP
			+ "Note: Most of the information contained in the Help screens of this program is also available in the various text files that came with this distribution (mainly the Readme.txt file)."
			;


		private static string HelpAbout =
			"MVShareStore Library example Server program. Version: " + Properties.Settings.Default.VERSION
			+ GAP
			+ "MVShareStore is a programmer's tool implemented as a MS Visual Studio C# Library / DLL that can be used by Windows applications to create, manipulate and share data programmatically within a single process as well as between disparate and separate processes running on the same host."
			+ GAP
			+ "This is an example Server program that can be used to implement the features of the Library and to create, modify and display any data you want the Library to handle. It can also be used in conjunction with the supplied example Client program."
			+ NL + "Please note that this does not demonstrate the way the MVShareStore Library must be used but simply one of the ways it can be."
			;
	}
}
