﻿using System;
using System.Windows.Forms;
using Microsoft.VisualBasic;
using MVShareStore;

namespace MVDataServer
{
	/// <summary>
	/// Handle all the DataShare related functions: Create, Inquire, Modify, Remove & Post.
	/// </summary>
	class DataShare
	{
		/// <summary>
		/// Create a new DataShare and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if a new DataShare was successfully created and stored, otherwise false</returns>
		internal static bool CreateDataShare(Main main) {
			string result = null;

			try {
				MVShareStore.DataShare dataShare = Data.CreateDataShare();
				if (dataShare == null) {
					if (main.showNotifications) MessageBox.Show("DataShare not created: " + Data.LastError, "Create New DataShare", MessageBoxButtons.OK);
					main.setStatusBar("DataShare not created: " + Data.LastError);
					return false;
				}
				else {
					result = "Successfully created new ";
					if (string.IsNullOrEmpty(dataShare.Name)) result += "Unnamed ";
					if(dataShare.ReadOnly) result += "ReadOnly ";
					if(dataShare.Private) result += "Private ";
					result += "DataShare. ID " + dataShare.ID;
					if (!string.IsNullOrEmpty(dataShare.Name)) result += ". Name = " + dataShare.Name;
					if (!string.IsNullOrEmpty(dataShare.Owner)) result += ". Owner = " + dataShare.Owner;
					result += ". Item count = " + dataShare.ItemCount;

					if (main.showNotifications) MessageBox.Show(result, "Create New DataShare", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(result);
					return true;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error creating new Data Share: " + (Data.LastError = e.Message), "Create Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error creating new Data Share: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Create a new DataShare using the currently active DataPack / DataShare / DataStore and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <param name="data">The DataPack / DataShare / DataStore to use to create the new DataShare</param>
		/// <returns>True if a new DataShare was successfully created and stored, otherwise false</returns>
		internal static bool CreateDataShare(Main main, MVShareStore.DataPack data) {
			string result = null;

			try {
				MVShareStore.DataShare dataShare = Data.CreateDataShare(data);
				if (dataShare == null) {
					if (main.showNotifications) MessageBox.Show("DataShare not created: " + Data.LastError, "Create New DataShare", MessageBoxButtons.OK);
					main.setStatusBar("DataShare not created: " + Data.LastError);
					return false;
				}
				else {
					result = "Successfully created new ";
					if (string.IsNullOrEmpty(dataShare.Name)) result += "Unnamed ";
					if (dataShare.ReadOnly) result += "ReadOnly ";
					if (dataShare.Private) result += "Private ";
					result += "DataShare. ID " + dataShare.ID;
					if (!string.IsNullOrEmpty(dataShare.Name)) result += ". Name = " + dataShare.Name;
					if (!string.IsNullOrEmpty(dataShare.Owner)) result += ". Owner = " + dataShare.Owner;
					result += " (using currently active " + data.GetType().ToString().Substring(13) + ")";
					result += ". Item count = " + dataShare.ItemCount;

					if (main.showNotifications) MessageBox.Show(result, "Create New DataShare", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(result);
					return true;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error creating new Data Share: " + (Data.LastError = e.Message), "Create Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error creating new Data Share: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Open an existing DataShare using its ID or its Name
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>The DataShare just opened or null if there was a problem</returns>
		internal static MVShareStore.DataShare OpenDataShare(Main main) {
			string name = null;
			long id = 0;

			try {
				string userinput = Interaction.InputBox("Please enter the ID or the Name of the DataShare you want to open", "Open Data Share", "1");
				long.TryParse(userinput, out id);
				if (id == 0) name = userinput;

				MVShareStore.DataShare dataShare = null;
				if (id > 0) dataShare = Data.OpenDataShare(id);
				else
					dataShare = Data.OpenDataShare(name);

				if(dataShare != null) {
					if (main.showNotifications) MessageBox.Show("Successfully Opened Data Share " + userinput + ". Item count = " + dataShare.ItemCount, "Open Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("Successfully Opened Data Share " + userinput + ". Item count = " + dataShare.ItemCount);
					return Data.CurrentDataShare = dataShare;
				}
				else {
					if (main.showNotifications) MessageBox.Show("Unable to Open the Data Share " + userinput + " Error: " + Data.LastError, "Open Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to Open the Data Share " + userinput + " Error: " + Data.LastError);
					return null;
				}
			}
			catch(Exception e) {
				if (main.showNotifications) MessageBox.Show("Error opening Data Share: " + (Data.LastError = e.Message), "Open Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error opening Data Share: " + (Data.LastError = e.Message));
				return null;
			}
		}

		/// <summary>
		/// List and optionally filter all the available DataShares
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if Shares were found and listed. False if there was a problem</returns>
		internal static bool ListDataShares(Main main) {
			int count = 0;
			bool filter = false, sort = false, edit = false;

			try {
				if (MessageBox.Show(main.setStatusBar("Do you want to allow editing of the Data Shares that are Listed?"), "List Data Shares", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					edit = true;
/*
				if (MessageBox.Show(main.setStatusBar("Do you want to filter which Data Shares are displayed?"), "List Data Shares", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					filter = true;

				if (MessageBox.Show(main.setStatusBar("Do you want to sort the order in which the Data Shares are displayed?"), "List Data Shares", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					sort = true;
*/
				count = Data.ListDataShares(edit, true, filter, sort).Count;

				main.setStatusBar("Successfully listed Data Shares. " + count + " DataShares found");
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error listing Data Shares: " + (Data.LastError = e.Message), "List Data Shares", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error listing Data Shares: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Edit the Data Items in the currently active Data Share and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully edited, otherwise false</returns>
		internal static bool EditDataShare(Main main) {
			try {
				//	We don't have a current Data Share so create a new one instead
				if (Data.CurrentDataShare == null) {
					if (MessageBox.Show(main.setStatusBar("No current Data Share available. Do you want to create a new one?"), "Edit Data Share", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						return CreateDataShare(main);
					else {
						if (main.showNotifications) MessageBox.Show("Unable to edit DataShare: No current Data Share available", "Edit Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
						main.setStatusBar(Data.LastError = "Unable to edit DataShare: No current Data Share available");
						return false;
					}
				}

				Data.LastError = null;
				MVShareStore.DataShare dataShare = Data.EditDataShare();
				if (!string.IsNullOrEmpty(Data.LastError)) {
					if (main.showNotifications) MessageBox.Show(Data.LastError, "Edit Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Successfully edited Data Share. Item count now = " + dataShare.ItemCount, "Edit Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Successfully edited Data Share. Item count now = " + dataShare.ItemCount);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error editing Data Share: " + (Data.LastError = e.Message), "Edit Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error editing Data Share: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// List all the Data Items in the currently active Data Share and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully displayed, otherwise false</returns>
		internal static bool ListDataShare(Main main) {
			try {
				//	We don't have a current Data Share so create a new one instead
				if (Data.CurrentDataShare == null) {
					if (MessageBox.Show(main.setStatusBar("No current Data Share available. Do you want to create a new one?"), "List Data Share", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						return CreateDataShare(main);
					else {
						if (main.showNotifications) MessageBox.Show("Unable to list the DataShare: No current Data Share available", "List Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
						main.setStatusBar(Data.LastError = "Unable to list the DataShare: No current Data Share available");
						return false;
					}
				}

				Data.ListDataShare();
				return true;
			}
			catch (Exception e) {
				MessageBox.Show("Error listing Data Share Items: " + (Data.LastError = e.Message), "List Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error listing Data Share Items: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Sort the contents of the currently active DataShare and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully sorted, otherwise false</returns>
		internal static bool SortDataShare(Main main) {
			try {
				if ((Data.CurrentDataShare == null) || (Data.CurrentDataShare.ItemCount == 0)) {
					if (main.showNotifications) MessageBox.Show("No current DataShare or no Data Items to sort in it", "Sort Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				bool descending = false;
				if (MessageBox.Show("Do you want to sort the Data Items into descending order?", "Sort Data Share Items", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) descending = true;
				bool sortByCode = false;
				if (MessageBox.Show("Do you want to sort the Data Items by their Code Field first?", "Sort Data Share Items", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) sortByCode = true;

				if (Data.SortDataPackItems(Data.CurrentDataShare, descending, sortByCode) == null) {
					if (main.showNotifications) MessageBox.Show(Data.LastError, "Sort Data Share Items", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(Data.LastError);
					return false;
				}

				Data.ListDataShare();
				main.setStatusBar("Data Items in the currently active Data Share sorted");
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error sorting DataShare contents: " + (Data.LastError = e.Message), "Sort DataShare Items", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error sorting DataShare contents: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Merge the supplied DataPack / DataShare / DataStore with another DataPack / DataShare / DataStore or a new DataPack or a DataStore off disk opened as a DataPack
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully merged, otherwise false</returns>
		internal static bool MergeDataShare(Main main) {
			try {
				if ((Data.CurrentDataShare == null) || (Data.CurrentDataShare.ItemCount == 0)) {
					if (main.showNotifications) MessageBox.Show("No current DataShare or no Data Items to merge in it", "Merge Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				//	TODO - This needs to be tested with a null DataPack and cancelled merge
				MVShareStore.DataShare dataShare = Data.CurrentDataShare;
				if ((dataShare = (MVShareStore.DataShare)Data.ShowMergeDataItemsForm(Data.CurrentDataShare)) == null) {
					if (main.showNotifications) MessageBox.Show(Data.LastError, "Merge Data Share Items", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(Data.LastError);
					return false;
				}

				Data.CurrentDataShare = dataShare;
				main.setStatusBar("Data Items in the currently active Data Share merged");
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error merging DataShare Data Items: " + (Data.LastError = e.Message), "Merge DataShare Items", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error merging DataShare Data Items: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Extract a DataPack Data Item from the currently active DataShare optionally removing it in the process
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the Data Pack was found and displayed for editing, otherwise false</returns>
		internal static bool ExtractDataPack(Main main) {
			string name = null;
			bool remove = false;

			try {
				if ((Data.CurrentDataShare == null) || (Data.CurrentDataShare.ItemCount == 0)) {
					if (main.showNotifications) MessageBox.Show("No current DataShare or no Data Items to show in it", "Extract Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("No current DataShare or no Data Items to show in it");
					return false;
				}

				name = Interaction.InputBox("Enter the name of the DataPack Data Item you want to extract (nested DataPacks can be accessed using the path / character in the name)", "Enter DataPack Data Item Name", "");
				if (string.IsNullOrEmpty(name)) {
					if (main.showNotifications) MessageBox.Show("No DataPack Data Item name specified", "Extract Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				DialogResult result = MessageBox.Show("Do you want to remove the DataPack Data Item after it has been extracted?", "Extract Data Pack", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
				if (result == DialogResult.Cancel) {
					if (main.showNotifications) MessageBox.Show("Extract of DataPack cancelled by user", "Extract Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar(Data.LastError = "Extract of Data Pack cancelled by user");
					return false;
				}
				else if (result == DialogResult.Yes) remove = true;

				MVShareStore.DataPack dataPack = Data.ExtractDataPack(Data.CurrentDataShare, name, remove);
				if (dataPack == null) {
					if (main.showNotifications) MessageBox.Show("Unable to extract the DataPack Data Item " + name + " Error: " + Data.LastError, "Extract Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to extract the DataPack Data Item " + name + " Error: " + Data.LastError);
					return false;
				}

				MVShareStore.forms.EditDataItems editDataItems = new MVShareStore.forms.EditDataItems(dataPack);
				editDataItems.ShowDialog();
//				Data.ShowEditDataItemsForm(dataPack);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error extracting DataPack Data Item: " + (Data.LastError = e.Message), "Extract Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error extracting DataPack Data Item: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Find a Data Item in the DataShare using its Name or Index
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <param name="edit">Edit the Data Item after it has been found? (defaults to false)</param>
		/// <returns>True if the Data Item was found, otherwise false</returns>
		internal static bool FindDataItem(Main main, bool edit = false) {
			DataItem item = null;
			string indexname = null;
			int index = -1;

			try {
				if ((Data.CurrentDataShare == null) || (Data.CurrentDataShare.ItemCount == 0)) {
					if (main.showNotifications) MessageBox.Show("No current DataShare or no Data Items to show in it", "Find / Edit Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				DialogResult result = MessageBox.Show("Do you want to search for / find the Data Item by Name? Click Yes if you do otherwise No for using its Index instead", "Find / Edit Data Item", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
				if (result == DialogResult.Cancel) {
					if (main.showNotifications) MessageBox.Show("Find / Edit of Data Item cancelled by user", "Find / Edit Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar(Data.LastError = "Find / Edit of Data Item cancelled by user");
					return false;
				}
				else if (result == DialogResult.No) {
					indexname = Interaction.InputBox("Enter the index / position of the Data Item you want to find / edit", "Enter Data Item Index", "0");
					index++;
					if (int.TryParse(indexname, out index))
						item = Data.findDataShareItem(index);
				}
				else if (result == DialogResult.Yes) {
					indexname = Interaction.InputBox("Enter the name of the Data Item you want to find / edit", "Enter Data Item Name", "");
					item = Data.findDataShareItem(indexname);
				}

				if (item == null) {
					if (main.showNotifications) MessageBox.Show("Unable to find the Data Item " + indexname + " Error: " + Data.LastError, "Find / Edit Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to find the Data Item " + indexname + " Error: " + Data.LastError);
					return false;
				}

				if (edit) {
					MVShareStore.forms.AddDataItem editDataItem = new MVShareStore.forms.AddDataItem(Data.CurrentDataShare, item);
					if (editDataItem.ShowDialog() == DialogResult.OK) {
						item = editDataItem.DataItem;
						if (index > -1) Data.CurrentDataShare.Edit(index, item);
						else
							Data.CurrentDataShare.Edit(indexname, item);
					}
				}

				if (main.showNotifications) MessageBox.Show("Data Item found / edited. Type = " + item.Type + " Name = " + item.Name + " Value = " + item.Value, "Find / Edit Data Item", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Data Item found / edited. Type = " + item.Type + " Name = " + item.Name + " Value = " + item.Value);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error finding / editing Data Item: " + (Data.LastError = e.Message), "Find / Edit Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error finding / editing Data Item: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Add new Header Data Items in the currently active Data Share and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully edited, otherwise false</returns>
		internal static bool AddShareHeaderData(Main main) {
			try {
				//	We don't have a current Data Share so create a new one instead
				if (Data.CurrentDataShare == null) {
					if (MessageBox.Show(main.setStatusBar("No current Data Share available. Do you want to create a new one?"), "Add Header Data Share", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.No) { 
						if (main.showNotifications) MessageBox.Show("Unable to add Header Data Items to the DataShare: No current Data Share available", "Add Header Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
						main.setStatusBar(Data.LastError = "Unable to add Header Data Items to the DataShare: No current Data Share available");
						return false;
					}
				}

				MVShareStore.DataShare dataShare = Data.AddDataShareHeaderData();

				if (dataShare.DataHeader != null) {
					if (main.showNotifications) MessageBox.Show("Successfully added Header Data Items to the Data Share. Header Data Item Count = " + dataShare.DataHeader.Data.ItemCount, "Add Header Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("Successfully added Header Data Items to the Data Share. Header Data Item Count = " + dataShare.DataHeader.Data.ItemCount);
				}
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error adding Header Data Items to the Data Share: " + (Data.LastError = e.Message), "Add Header Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error adding Header Data Items to the Data Share: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// List all the Data Header Items in the currently active Data Share and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare Header Items were successfully displayed, otherwise false</returns>
		internal static bool ListShareHeaderData(Main main) {
			try {
				//	We don't have a current Data Share so create a new one instead
				if (Data.CurrentDataShare == null) {
					if (MessageBox.Show(main.setStatusBar("No current Data Share available. Do you want to create a new one?"), "List Data Share Header Items", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						CreateDataShare(main);
					else {
						if (main.showNotifications) MessageBox.Show("Unable to list the DataShare: No current Data Share available", "List Data Share Header Items", MessageBoxButtons.OK, MessageBoxIcon.Error);
						main.setStatusBar(Data.LastError = "Unable to list the Header Data Items in the DataShare: No current Data Share available");
						return false;
					}
				}

				Data.LastError = null;
				Data.ListDataShareHeaderData();

				if (!string.IsNullOrEmpty(Data.LastError)) {
					if (main.showNotifications) MessageBox.Show(Data.LastError, "List Data Share Header Items", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(Data.LastError);
					return false;
				}
				return true;
			}
			catch (Exception e) {
				MessageBox.Show("Error listing Data Share Header Items: " + (Data.LastError = e.Message), "List Data Share Header Items", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error listing Data Share Header Items: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Close the currently active DataShare and remove it from memory if we are the last user (this will invalidate the current DataShare in the Data class)
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully closed, otherwise false</returns>
		internal static bool CloseDataShare(Main main) {
			string name = null;

			try {
				if (Data.CurrentDataShare == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataShare to close", "Close Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				name = Data.CurrentDataShare.Name;
				if (Data.CloseDataShare()) {
					if (main.showNotifications) MessageBox.Show("Our copy of the currently active Data Share " + name + " is now closed", "Close Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("Our copy of the currently active Data Share " + name + " is now closed");
					return true;
				}
				else {
					main.setStatusBar("There was a problem closing the currently active Data Share " + name + ": " + Data.LastError);
					return false;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error closing DataShare " + name + ": " + (Data.LastError = e.Message), "Close Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error closing DataShare " + name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Convert the currently active DataShare into a new DataStore
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully converted into a DataStore, otherwise false</returns>
		internal static bool ConvertDataShareToStore(Main main) {
			bool showDialog = false;

			try {
				if (MessageBox.Show("Do you want to enter additional information regarding the new DataStore?", "Convert to DataStore", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					showDialog = true;

				if (Data.ConvertDataShareToDataStore(null, showDialog) == null) {
					if (main.showNotifications) MessageBox.Show("Error converting DataShare " + Data.CurrentDataShare.Name + " to a DataStore: " + Data.LastError, "Convert DataShare to DataStore", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Error converting DataShare " + Data.CurrentDataShare.Name + " to a DataStore: " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Currently active Data Share now converted into a new Data Store " + Data.CurrentDataStore.Name, "Convert DataShare to DataStore", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Currently active Data Share now converted into a new Data Store " + Data.CurrentDataStore.Name);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error converting DataShare to a DataStore: " + Data.CurrentDataShare.Name + ": " + (Data.LastError = e.Message), "Convert DataShare to DataStore", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error converting DataShare to a DataStore: " + Data.CurrentDataShare.Name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Post / Publish the currently active DataShare
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully posted / published, otherwise false</returns>
		internal static bool PostDataShare(Main main) {
			int index = -1;

			try {
				if((index = Data.PostDataShare(null, true)) == -1) {
					if (main.showNotifications) MessageBox.Show("Error Posting / Publishing DataShare " + Data.CurrentDataShare.Name + ": " + Data.LastError, "Post / Publish DataShare", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Error Posting / Publishing DataShare " + Data.CurrentDataShare.Name  + ": " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Currently active Data Share " + Data.CurrentDataShare.Name + " now successfully posted / published. Header List Index = " + index, "Post / Publish DataShare", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Currently active Data Share " + Data.CurrentDataShare.Name + " now posted / published. Header List Index = " + index);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error Posting / Publishing DataShare " + Data.CurrentDataShare.Name + ": " + (Data.LastError = e.Message), "Post / Publish DataShare", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error Posting / Publishing DataShare: " + Data.CurrentDataShare.Name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Repost / update the currently active DataShare
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully posted / published, otherwise false</returns>
		internal static bool RepostDataShare(Main main) {
			int index = -1;

			try {
				if ((index = Data.RepostDataShare(null)) == -1) {
					if (main.showNotifications) MessageBox.Show("Error Reposting DataShare " + Data.CurrentDataShare.Name + ": " + Data.LastError, "Repost DataShare", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Error Reposting DataShare " + Data.CurrentDataShare.Name + ": " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Currently active Data Share " + Data.CurrentDataShare.Name + " now successfully reposted. Header List Index = " + index, "Repost DataShare", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Currently active Data Share " + Data.CurrentDataShare.Name + " now successfully reposted. Header List Index = " + index);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error Reposting DataShare " + Data.CurrentDataShare.Name + ": " + (Data.LastError = e.Message), "Repost DataShare", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error Reposting DataShare: " + Data.CurrentDataShare.Name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Set the Code Field Header / Description to a value other than 'Code' to describe it's usage in this instance
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare Code Description was changed to a new value, otherwise false</returns>
		internal static bool SetDataShareCodeDescription(Main main) {
			try {
				if (Data.CurrentDataShare == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataShare to set code description for", "Set Code Field Description", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				string description = Interaction.InputBox("Enter the Header / Description of the Code Field in this DataShare", "Set Code Field Description", Data.CurrentDataShare.CodeDescription);
				if (!string.IsNullOrEmpty(description)) {
					Data.CurrentDataShare.CodeDescription = description;
					if (main.showNotifications) MessageBox.Show("Currently active Data Share Code Field Description now set to " + Data.CurrentDataShare.CodeDescription, "Set Code Field Description", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Currently active Data Share Field Description now set to " + Data.CurrentDataShare.CodeDescription);
					return true;
				}
				return false;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error Setting DataShare Code Field Description: " + (Data.LastError = e.Message), "Set Code Field Description", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error Setting DataShare Code Field Description: " + (Data.LastError = e.Message));
				return false;
			}
		}
	}
}
