﻿using System;
using System.Windows.Forms;
using Microsoft.VisualBasic;
using MVShareStore;

namespace MVDataClient
{
	/// <summary>
	/// Handle all the DataStore related functions: Create, Inquire, Modify, Remove & Post.
	/// </summary>
	class DataStore
	{
		/// <summary>
		/// Create a new DataStore and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if a new DataStore was successfully created and stored, otherwise false</returns>
		internal static bool CreateDataStore(Main main) {
			try {
				MVShareStore.DataStore dataStore = Data.CreateDataStore();
				if (dataStore == null) {
					if (main.showNotifications) MessageBox.Show("DataStore not created: " + Data.LastError, "Create New DataStore", MessageBoxButtons.OK);
					main.setStatusBar("DataStore not created: " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Successfully created new DataStore " + dataStore.Name + ". Item count = " + dataStore.ItemCount, "Create New DataStore", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Successfully created new DataStore " + dataStore.Name + ". Item count = " + dataStore.ItemCount);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error creating new Data Store: " + (Data.LastError = e.Message), "Create New DataStore", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error creating new Data Store: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Create a new DataStore using the currently active DataPack / DataShare / DataStore and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <param name="data">The DataPack / DataShare / DataStore to use to create the new DataShare</param>
		/// <returns>True if a new DataStore was successfully created and stored, otherwise false</returns>
		internal static bool CreateDataStore(Main main, MVShareStore.DataPack data) {
			string result = null;

			try {
				MVShareStore.DataStore dataStore = Data.CreateDataStore(data);
				if (dataStore == null) {
					if (main.showNotifications) MessageBox.Show("DataStore not created: " + Data.LastError, "Create New DataStore", MessageBoxButtons.OK);
					main.setStatusBar("DataStore not created: " + Data.LastError);
					return false;
				}
				else {
					result = "Successfully created new DataStore: " + dataStore.Name;
					if (!string.IsNullOrEmpty(dataStore.Path)) result += ". Location: " + dataStore.Path;
					result += " (using currently active " + data.GetType().ToString().Substring(13) + ")";
					result += ". Item count = " + dataStore.ItemCount;

					if (main.showNotifications) MessageBox.Show(result, "Create New DataStore", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(result);
					return true;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error creating new Data Store: " + (Data.LastError = e.Message), "Create Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error creating new Data Store: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Edit the Data Items in the currently active Data Store and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully edited, otherwise false</returns>
		internal static bool EditDataStore(Main main) {
			try {
				//	We don't have a current Data Share so create a new one instead
				if (Data.CurrentDataStore == null) {
					if (MessageBox.Show(main.setStatusBar("No current Data Store available. Do you want to create a new one?"), "Edit Data Store", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						return CreateDataStore(main);
					else {
						if (main.showNotifications) MessageBox.Show("Unable to edit DataStore: No current Data Store available", "Edit Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
						main.setStatusBar(Data.LastError = "Unable to edit DataStore: No current Data Store available");
						return false;
					}
				}

				Data.LastError = null;
				MVShareStore.DataStore dataStore = Data.EditDataStore();
				if (!string.IsNullOrEmpty(Data.LastError)) {
					if (main.showNotifications) MessageBox.Show(Data.LastError, "Edit Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar(Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Successfully edited Data Store " + Data.CurrentDataStore.Name + ". Item count now = " + dataStore.ItemCount, "Edit Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("Successfully edited Data Store " + Data.CurrentDataStore.Name + ". Item count now = " + dataStore.ItemCount);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error editing Data Store: " + (Data.LastError = e.Message), "Edit Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error editing Data Store: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// List the Data Items in the currently active Data Store and return true if we were successful otherwise false
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully displayed, otherwise false</returns>
		internal static bool ListDataStore(Main main) {
			try {
				//	We don't have a current Data Store so create a new one instead
				if (Data.CurrentDataStore == null) {
					if (MessageBox.Show(main.setStatusBar("No current Data Store available. Do you want to create a new one?"), "List Data Store", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						return CreateDataStore(main);
					else {
						if (main.showNotifications) MessageBox.Show("Unable to list DataStore: No current Data Store available", "List Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
						main.setStatusBar(Data.LastError = "Unable to list DataStore: No current Data Store available");
						return false;
					}
				}

				Data.ListDataStore();
				return true;
			}
			catch (Exception e) {
				MessageBox.Show("Error listing Data Store Items: " + (Data.LastError = e.Message), "List Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error listing Data Store Items: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Find a Data Item in the DataStore using its Name or Index
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the Data Item was found, otherwise false</returns>
		internal static bool FindDataItem(Main main) {
			DataItem item = null;
			string indexname = null;

			try {
				if ((Data.CurrentDataStore == null) || (Data.CurrentDataStore.ItemCount == 0)) {
					if (main.showNotifications) MessageBox.Show("No current DataStore or no Data Items to show in it", "Find Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
					return false;
				}

				DialogResult result = MessageBox.Show("Do you want to search for / find the Data Item by Name? Click Yes if you do otherwise No for using its Index instead", "Find Data Item", MessageBoxButtons.YesNoCancel, MessageBoxIcon.Question);
				if (result == DialogResult.Cancel) {
					if (main.showNotifications) MessageBox.Show("Find of Data Item cancelled by user", "Find Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar(Data.LastError = "Find of Data Item cancelled by user");
					return false;
				}
				else if (result == DialogResult.No) {
					indexname = Interaction.InputBox("Enter the index / position of the Data Item you want to find", "Enter Data Item Index", "0");
					int index = 0;
					if(int.TryParse(indexname, out index))
						item = Data.findDataStoreItem(index);
				}
				else if (result == DialogResult.Yes) {
					indexname = Interaction.InputBox("Enter the name of the Data Item you want to find", "Enter Data Item Name", "");
					item = Data.findDataStoreItem(indexname);
				}

				if (item == null) {
					if (main.showNotifications) MessageBox.Show("Unable to find the Data Item " + indexname + " Error: " + Data.LastError, "Find Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to find the Data Item " + indexname + " Error: " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("Data Item found. Type = " + item.Type + " Name = " + item.Name + " Value = " + item.Value, "Find Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar("Data Item found. Type = " + item.Type + " Name = " + item.Name + " Value = " + item.Value);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error finding Data Item: " + (Data.LastError = e.Message), "Find Data Item", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error finding Data Item: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Close the currently active DataStore, optionally saving it first if required (this will invalidate the current DataStore in the Data class). However, if it has been saved it will remain on disk.
		/// </summary>	
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully closed, otherwise false</returns>
		internal static bool CloseDataStore(Main main) {
			string name = null;
			bool save = false;

			try {
				if (Data.CurrentDataStore == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataStore to close", "Close Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("No currently active DataStore to close");
					return false;
				}

				if(Data.CurrentDataStore.Dirty) {
					if (MessageBox.Show(main.setStatusBar("The current Data Store " + Data.CurrentDataStore.Name + " has been changed since the last save. Do you want to save it before closing?"), "Save Data Store", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						save = true;
				}

				name = Data.CurrentDataStore.Name;
				if (Data.CloseDataStore(Data.CurrentDataStore, save)) {
					if (save) {
						if (main.showNotifications) MessageBox.Show("The currently active Data Store " + name + " has been successfully saved and closed", "Close Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
						main.setStatusBar("The currently active Data Store " + name + " has been successfully saved and closed");
					}
					else {
						if (main.showNotifications) MessageBox.Show("The currently active Data Store " + name + " is now closed", "Close Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
						main.setStatusBar("The currently active Data Store " + name + " is now closed");
					}
					return true;
				}
				else {
					if (main.showNotifications) MessageBox.Show("Unable to close the currently active DataStore " + name + ": " + Data.LastError, "Close Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to close the currently active DataStore " + name + ": " + Data.LastError);
					return false;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error closing DataStore " + name + ": " + (Data.LastError = e.Message), "Close Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error closing DataStore " + name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Delete the currently active DataStore and remove it from memory and disk (this will invalidate the current DataStore in the Data class)
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully deleted, otherwise false</returns>
		internal static bool DeleteDataStore(Main main) {
			string name = null;

			try {
				if (Data.CurrentDataStore == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataStore to delete", "Delete Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("No currently active DataStore to delete");
					return false;
				}

				name = Data.CurrentDataStore.Name;
				if (Data.DeleteDataStore()) {
					if (main.showNotifications) MessageBox.Show("The currently active Data Store " + name + " has been successfully deleted", "Delete Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("The currently active Data Store " + name + " has been successfully deleted");
					return true;
				}
				else {
					if (main.showNotifications) MessageBox.Show("Unable to delete the currently active DataStore " + name + ": " + Data.LastError, "Delete Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to delete the currently active DataStore " + name + ": " + Data.LastError);
					return false;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error deleting DataShare " + name + ": " + (Data.LastError = e.Message), "Delete Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error deleting DataStore " + name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Convert the currently active DataStore into a new DataShare
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <param name="dataStore">The optional DataStore to convert to a DataShare</param>
		/// <returns>True if the DataStore was successfully converted into a DataShare, otherwise false</returns>
		internal static bool ConvertDataStoreToShare(Main main) {
			bool showDialog = false;

			try {
				if (MessageBox.Show("Do you want to enter additional information regarding the new DataShare?", "Convert DataStore to DataShare", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					showDialog = true;

				if (Data.ConvertDataStoreToDataShare(null, showDialog) == null) {
					if (main.showNotifications) MessageBox.Show("Error converting DataStore " + Data.CurrentDataStore.Name + " to a DataShare: " + Data.LastError, "Convert DataStore to DataShare", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Error converting DataStore " + Data.CurrentDataStore.Name + " to a DataShare: " + Data.LastError);
					return false;
				}

				main.setStatusBar("Data Store " + Data.CurrentDataStore.Name + " now converted into a new Data Share");
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error converting DataStore " + Data.CurrentDataStore.Name + " to a DataShare: " + (Data.LastError = e.Message), "Convert DataStore to DataShare", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error converting DataStore " + Data.CurrentDataStore.Name + " to a DataShare: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Open the specified DataStore and make it the currently active one, optionally saving the currently active one if required..
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully opened, otherwise false</returns>
		internal static bool OpenDataStore(Main main) {
			try {
				if ((Data.CurrentDataStore != null) && (Data.CurrentDataStore.Dirty)) {
					if (MessageBox.Show("Do you want to Save the changes you have made to the currently active DataStore " + Data.CurrentDataStore.Name + " first??", "Save Open Data Store", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
						Data.CurrentDataStore.Close(true, false);
				}

				if ((Data.CurrentDataStore = Data.OpenDataStore(null, null)) == null) {
					if (main.showNotifications) MessageBox.Show("Unable to open the DataStore: " + Data.LastError, "Open Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to open the DataStore: " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("The now currently active DataStore " + Data.CurrentDataStore.Name + " has now been opened from: " + Data.CurrentDataStore.FileName, "Open Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("The now currently active Data Store " + Data.CurrentDataStore.Name + " has now been has now been opened from: " + Data.CurrentDataStore.FileName);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error opening DataStore " + Data.CurrentDataStore.Name + ": " + e, "Open Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error opening DataStore " + Data.CurrentDataStore.Name + ": " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Open the specified DataStore and make it the currently active DataPack.
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully opened and made the currently active DataPack, otherwise false</returns>
		internal static bool OpenAsDataPack(Main main) {
			try {
				if ((Data.CurrentDataPack = Data.OpenAsDataPack(null, null)) == null) {
					if (main.showNotifications) MessageBox.Show("Unable to open the DataStore: " + Data.LastError, "Open Data Store as Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to open the DataStore: " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("The currently active DataPack has now been opened using DataStore " + Data.LastFile, "Open Data Store as Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("The currently active Data Pack has now been has now been opened using DataStore " + Data.LastFile);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error opening DataStore: " + e, "Open Data Store as Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error opening DataStore: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Open the specified DataStore and make it the currently active DataShare.
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully opened and made the		currently active DataShare, otherwise false</returns>
		internal static bool OpenAsDataShare(Main main) {
			bool showDialog = false;

			try {
				if (MessageBox.Show("Do you want to enter additional information about the DataShare using the create DataShare dialog?", "Open Data Store as Data Share", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					showDialog = true;

				if ((Data.CurrentDataShare = Data.OpenAsDataShare(null, null, showDialog)) == null) {
					if (main.showNotifications) MessageBox.Show("Unable to open the DataStore: " + Data.LastError, "Open Data Store as Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to open the DataStore: " + Data.LastError);
					return false;
				}

				if (main.showNotifications) MessageBox.Show("The currently active DataShare " + Data.CurrentDataShare.Name + " has now been opened using DataStore " + Data.LastFile, "Open Data Store as Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
				main.setStatusBar("The currently active DataShare " + Data.CurrentDataShare.Name + " has now been opened using DataStore " + Data.LastFile);
				return true;
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error opening DataStore: " + e, "Open Data Store as Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error opening DataStore: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Save the currently active DataStore to disk, optionally requesting a name and path
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataStore was successfully saved, otherwise false</returns>
		internal static bool SaveDataStore(Main main) {
			bool showDialog = false;

			try {
				if (Data.CurrentDataStore == null) {
					if (MessageBox.Show(main.setStatusBar("No currently active DataStore. Do you want to create a new one first?"), "Save Data Store", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
						CreateDataStore(main);
					}
				}

				if (Data.CurrentDataStore == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataStore to save", "Save Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar(Data.LastError = "No currently active DataStore to save");
					return false;
				}

				if (MessageBox.Show("Do you want to enter / modify the name and / or path information for the DataStore?", "Save Data Store", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					showDialog = true;

				if (Data.SaveDataStore(Data.CurrentDataStore, null, showDialog)) {
					if (main.showNotifications) MessageBox.Show("The currently active DataStore " + Data.CurrentDataStore.Name + " has now been saved as: " + Data.CurrentDataStore.FileName, "Save Data Store", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("The currently active Data Store " + Data.CurrentDataStore.Name + " has now been saved as: " + Data.CurrentDataStore.FileName);
					return true;
				}
				else {
					if (main.showNotifications) MessageBox.Show("Unable to save the currently active DataStore: " + Data.LastError, "Save Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to save the currently active DataStore: " + Data.LastError);
					return false;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error saving DataStore: " + (Data.LastError = e.Message), "Save Data Store", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error saving DataStore: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Save the currently active DataPack to disk, optionally requesting a name and path
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataPack was successfully saved, otherwise false</returns>
		internal static bool SaveDataPack(Main main) {

			try {
				if (Data.CurrentDataPack == null) {
					if (MessageBox.Show(main.setStatusBar("No currently active DataPack. Do you want to create a new one first?"), "Save Data Pack", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
						DataPack.CreateDataPack(main);
					}
				}

				if (Data.CurrentDataPack == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataPack to save", "Save Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar(Data.LastError = "No currently active DataPack to save");
					return false;
				}

//				if (MessageBox.Show("Do you want to enter name and / or path information for the DataStore of the DataPack?", "Save Data Pack", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
//
//				}

				if (Data.SaveDataPack(Data.CurrentDataPack, null)) {
					if (main.showNotifications) MessageBox.Show("The currently active DataPack has now been saved as " + Data.LastFile, "Save Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("The currently active Data Pack has now been saved as " + Data.LastFile);
					return true;
				}
				else {
					if (main.showNotifications) MessageBox.Show("Unable to save the currently active DataPack: " + Data.LastError, "Save Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to save the currently active DataPack: " + Data.LastError);
					return false;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error saving DataPack: " + (Data.LastError = e.Message), "Save Data Pack", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error saving DataPack: " + (Data.LastError = e.Message));
				return false;
			}
		}

		/// <summary>
		/// Save the currently active DataShare to disk, optionally requesting a name and path
		/// </summary>
		/// <param name="main">The Main application form</param>
		/// <returns>True if the DataShare was successfully saved, otherwise false</returns>	
		internal static bool SaveDataShare(Main main) {
			bool showDialog = false;

			try {
				if (Data.CurrentDataShare == null) {
					if (MessageBox.Show(main.setStatusBar("No currently active DataShare. Do you want to create a new one first?"), "Save Data Share", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes) {
						DataShare.CreateDataShare(main);
					}
				}

				if (Data.CurrentDataShare == null) {
					if (main.showNotifications) MessageBox.Show("No currently active DataShare to save", "Save Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar(Data.LastError = "No currently active DataShare to save");
					return false;
				}

				if (MessageBox.Show("Do you want to enter name and / or path information for the DataShare?", "Save Data Share", MessageBoxButtons.YesNo, MessageBoxIcon.Question) == DialogResult.Yes)
					showDialog = true;

				if (Data.SaveDataShare(Data.CurrentDataShare, null, showDialog)) {
					if (main.showNotifications) MessageBox.Show("The currently active DataShare " + Data.CurrentDataShare.Name + " has now been saved as " + Data.LastFile, "Save Data Share", MessageBoxButtons.OK, MessageBoxIcon.Information);
					main.setStatusBar("The currently active Data Share " + Data.CurrentDataShare.Name + " has now been saved as " + Data.LastFile);
					return true;
				}
				else {
					if (main.showNotifications) MessageBox.Show("Unable to save the currently active DataShare: " + Data.LastError, "Save Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
					main.setStatusBar("Unable to save the currently active DataShare: " + Data.LastError);
					return false;
				}
			}
			catch (Exception e) {
				if (main.showNotifications) MessageBox.Show("Error saving DataShare: " + (Data.LastError = e.Message), "Save Data Share", MessageBoxButtons.OK, MessageBoxIcon.Error);
				main.setStatusBar(Data.LastError = "Error saving DataShare: " + (Data.LastError = e.Message));
				return false;
			}
		}
	}
}
