﻿using MVShareStore;
using System;
using System.IO;
using MVConfig;

namespace MVConfigHandler {

	internal class ConvertFiles {

		private static readonly string XML_CONFIG_HEADER = "<!--Config / Ini file created by the MV Config File Library V::";

		/// <summary>
		/// The default file name extension to use for Config Files
		/// </summary>
		private static readonly string CONFIG_EXT = "mdc";

		/// <summary>
		/// The default file name extension for DataStores
		/// </summary>
//		private static readonly string DEFAULT_EXT = ".mds";

		/// <summary>
		/// Check to see if the specified Config file needs to be4 converted from the old XML to the new DataStore format?
		/// </summary>
		/// <param name="filename">The name of the Config file to check</param>
		/// <returns>True if the Config file is found and in the old XML format otherwise false if it is not found or is in the new DataStore format</returns>
		internal static bool ConvertRequred(string filename) {
			try {
				if (string.IsNullOrWhiteSpace(filename)) {
					Data.LastError = "No Config file name specified for file conversion check";
					return false;
				}

				if (!File.Exists(filename)) {
					Data.LastError = "Config file " + filename + " not found for file conversion check";
					return false;
				}

				//	Check to see if the expected Header comment is in the old XML format file?
				string[] lines = File.ReadAllLines(filename);
				if (lines[0].IndexOf(XML_CONFIG_HEADER) > -1) {
					_xmlLines = lines;
					return true;
				}
				return false;
			}
			catch (Exception e) {
				Data.LastError = "Exception checking to see if a conversion is required for XML Config File: " + filename + ": " + e.Message;
				return false;
			}
		}

		/// <summary>
		/// Read the MVConfig XML File on disk and add all it's entries into a DataStore which we then return
		/// </summary>
		/// <param name="filename">The name of the MVConfig file to open and load</param>
		/// <returns>All the entries in the MVConfiog file in a DataStore</returns>
		internal static DataStore ReadXMLFile(string filename) {
			DataStore store = new DataStore();

			try {
				//	Make sure we can open and load the file and that there is something in it.
				_xmlConfig = new MVConfig.MVConfigFile(filename);
				if ((_xmlConfig == null) || (!_xmlConfig.getLoaded()) || (_xmlConfig.getCount() == 0)) {
					Data.LastError = "Unable to read MVConfig XML file: " + MVConfig.MVConfigFile.getLastError();
					return null;
				}

				//	Now read up all the Global entries
				MVConfigEntry[] entries = _xmlConfig.getEntries();
				if ((entries != null) && (entries.Length > 0)) {
					foreach (MVConfigEntry entry in entries) {
						store = addConfigEntry(store, entry);
					}
				}

				//	Now do the same for any Sections.
				string[] sections = _xmlConfig.getSectionNames();
				if (sections == null || sections.Length == 0) return store;
				foreach (string section in sections) {
					entries = _xmlConfig.getSectionEntries(section);
					if ((entries != null) && (entries.Length > 0)) {
						DataStore dataStore = new DataStore();
						foreach (MVConfigEntry entry in entries) {
							dataStore = addConfigEntry(dataStore, entry);
						}
						//						store.addDataItem(new DataItem(dataStore, SECTION_NAME + section));
						store.addDataItem(new DataItem(dataStore, section));
					}
				}

				return store;
			}
			catch (Exception e) {
				Data.LastError = "Exception reading the XML Config File: " + filename + ": " + e.Message;
				return null;
			}
		}

		internal static DataStore addConfigEntry(DataStore store, MVConfigEntry entry) {
			DataItem item = null;
			if ((store == null) || (entry == null)) return null;
			if (entry.isString()) item = new DataItem(entry.Value, entry.Name);
			else if (entry.isBoolean()) item = new DataItem(entry.BoolValue, entry.Name);
			else if (entry.isInteger()) item = new DataItem(entry.IntValue, entry.Name);
			store.addDataItem(item, true);

			return store;
		}

		internal static bool validFile(string filename) {
			if (string.IsNullOrEmpty(filename)) return false;
			if (!System.IO.Path.HasExtension(filename))
				filename = System.IO.Path.ChangeExtension(filename, CONFIG_EXT);
			if (!File.Exists(filename)) return false;
			return true;
		}

		private static MVConfig.MVConfigFile _xmlConfig = null;
		private static string[] _xmlLines = null;
	}
}
